<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;

class PolygonService
{
    protected string $apiKey;
    protected string $baseUri;

    public function __construct()
    {
        $this->apiKey  = config('polygon.key');
        $this->baseUri = rtrim(config('polygon.base_uri'), '/');
    }

    /**
     * Perform a GET request to the Polygon API.
     *
     * @param string $endpoint
     * @param array  $params
     * @return array
     * @throws \Exception
     */
    protected function get(string $endpoint, array $params = []): array
    {
        // Merge apiKey into parameters
        $params['apiKey'] = $this->apiKey;

        // Send GET request
        $response = Http::get("{$this->baseUri}{$endpoint}", $params);

        if (! $response->successful()) {
            throw new \Exception('Polygon API Error: ' . $response->body());
        }

        return $response->json();
    }

    /**
     * Fetch assets for a given category slug using Polygon.io reference endpoints.
     *
     * @param string $slug
     * @return array
     * @throws \InvalidArgumentException
     * @throws \Exception
     */
    public function getAssetsByCategorySlug(string $slug): array
    {
        $mapping = match ($slug) {
            'stock' => [
                '/v3/reference/tickers',
                [
                    'market' => 'stocks',
                    'active' => 'true',
                    'sort'   => 'ticker',
                    'order'  => 'asc',
                    'limit'  => 1000,
                ],
            ],

            'index' => [
                '/v3/reference/tickers',
                [
                    'type'   => 'INDEX',
                    'active' => 'true',
                    'sort'   => 'ticker',
                    'order'  => 'asc',
                    'limit'  => 1000,
                ],
            ],

            'etfs' => [
                '/v3/reference/tickers',
                [
                    'type'   => 'ETF',
                    'active' => 'true',
                    'sort'   => 'ticker',
                    'order'  => 'asc',
                    'limit'  => 1000,
                ],
            ],

            'crypto' => [
                '/v3/reference/tickers',
                [
                    'market' => 'crypto',
                    'active' => 'true',
                    'sort'   => 'ticker',
                    'order'  => 'asc',
                    'limit'  => 1000,
                ],
            ],

            'forex' => [
                '/v3/reference/tickers',
                [
                    'market' => 'fx',
                    'active' => 'true',
                    'sort'   => 'ticker',
                    'order'  => 'asc',
                    'limit'  => 1000,
                ],
            ],

            'real-estate' => [
                '/v3/reference/tickers',
                [
                    'search' => 'REIT',
                    'active' => 'true',
                    'sort'   => 'ticker',
                    'order'  => 'asc',
                    'limit'  => 1000,
                ],
            ],

            // Polygon.io doesn't officially support commodities, use broad search
            'commodities' => [
                '/v3/reference/tickers',
                [
                    'search' => 'COM',  // generic commodity tag search
                    'active' => 'true',
                    'sort'   => 'ticker',
                    'order'  => 'asc',
                    'limit'  => 100,
                ],
            ],

            default => throw new \InvalidArgumentException("Polygon API does not support Annuities Private Equity and Treasury Bonds"),
        };

        [$endpoint, $params] = $mapping;
        $data = $this->get($endpoint, $params);

        return $data['results'] ?? $data['tickers'] ?? [];
    }
}