<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class AlphaVantageService
{
    protected string $apiKey;

    public function __construct()
    {
        $this->apiKey = config('services.alphavantage.key');

        if (empty($this->apiKey)) {
            throw new \Exception('Alpha Vantage API key is not set.');
        }
    }

    /**
     * Fetch real-time price for any ticker based on category slug.
     */
    public function getPriceByCategory(string $ticker, string $category)
    {
        switch ($category) {
            case 'crypto':
                return $this->getCryptoPrice($ticker);
            case 'fx':
                return $this->getFxPrice($ticker);
            case 'stocks':
            case 'otc':
                return $this->getStockPrice($ticker);
            case 'indices':
                return $this->getIndexPrice($ticker);
            default:
                return null;
        }
    }

    /**
     * Fetch full ticker details: overview, real-time price, daily & intraday series.
     * This method is independent of existing price functions.
     * @param string $ticker Polygon-style ticker
     * @param string $category Category slug
     * @param string $intradayInterval e.g. '60min'
     * @param bool $fullDaily
     * @param bool $fullIntraday
     * @return array{
     *    overview:?array,
     *    price:float|null,
     *    daily:?array,
     *    intraday:?array
     * }
     */
    public function getFullTickerDetails(
        string $ticker,
        string $category,
        string $intradayInterval = '60min',
        bool $fullDaily = false,
        bool $fullIntraday = false
    ): array {
        // Determine symbol for Alpha Vantage
        switch ($category) {
            case 'crypto':
                [$from, $to] = $this->splitTicker($ticker, 'X:');
                $symbol = [$from, $to];
                // Crypto uses FX endpoints for price
                break;
            case 'fx':
                [$from, $to] = $this->splitTicker($ticker, 'C:');
                $symbol = [$from, $to];
                break;
            case 'indices':
                $symbol = Str::after($ticker, 'I:');
                break;
            default:
                // stocks, otc
                $symbol = $ticker;
                break;
        }

        // Overview only for equities (stocks, otc, indices)
        $overview = null;
        if (is_string($symbol)) {
            $overview = $this->fetchOverview($symbol);
        }

        // Price using existing logic
        $priceData = $this->getPriceByCategory($ticker, $category);

        // Historical series
        $daily = is_string($symbol)
            ? $this->fetchDailySeries($symbol, $fullDaily)
            : null;

        $intraday = is_string($symbol)
            ? $this->fetchIntradaySeries($symbol, $intradayInterval, $fullIntraday)
            : null;

        return [
            'overview'  => $overview,
            'price'     => $priceData,
            'daily'     => $daily,
            'intraday'  => $intraday,
        ];
    }

    // --- Existing protected methods ---

    protected function getCryptoPrice(string $ticker)
    {
        [$from, $to] = $this->splitTicker($ticker, 'X:');
        $data = Http::get('https://www.alphavantage.co/query', [
            'function'      => 'CURRENCY_EXCHANGE_RATE',
            'from_currency' => $from,
            'to_currency'   => $to,
            'apikey'        => $this->apiKey,
        ])->json();

        return [
            'ctitle' => "{$from}/{$to}",
            'price'  => $data['Realtime Currency Exchange Rate']['5. Exchange Rate'] ?? null,
        ];
    }

    protected function getFxPrice(string $ticker)
    {
        [$from, $to] = $this->splitTicker($ticker, 'C:');
        $data = Http::get('https://www.alphavantage.co/query', [
            'function'      => 'CURRENCY_EXCHANGE_RATE',
            'from_currency' => $from,
            'to_currency'   => $to,
            'apikey'        => $this->apiKey,
        ])->json();

        return [
            'ctitle' => "{$from}/{$to}",
            'price'  => $data['Realtime Currency Exchange Rate']['5. Exchange Rate'] ?? null,
        ];
    }

    protected function getStockPrice(string $ticker)
    {
        $data = Http::get('https://www.alphavantage.co/query', [
            'function' => 'GLOBAL_QUOTE',
            'symbol'   => $ticker,
            'apikey'   => $this->apiKey,
        ])->json();

        return ['price' => $data['Global Quote']['05. price'] ?? null];
    }

    protected function getIndexPrice(string $ticker)
    {
        $symbol = Str::after($ticker, 'I:');
        $data = Http::get('https://www.alphavantage.co/query', [
            'function' => 'GLOBAL_QUOTE',
            'symbol'   => $symbol,
            'apikey'   => $this->apiKey,
        ])->json();

        return [
            'ctitle' => $symbol,
            'price'  => $data['Global Quote']['05. price'] ?? null,
        ];
    }

    protected function splitTicker(string $ticker, string $prefix): array
    {
        $clean = Str::after($ticker, $prefix);
        $to = substr($clean, -3);
        $from = substr($clean, 0, -3);
        return [$from, $to];
    }

    /**
     * Fetch company overview metadata.
     */
    protected function fetchOverview(string $symbol): ?array
    {
        $response = Http::get('https://www.alphavantage.co/query', [
            'function' => 'OVERVIEW',
            'symbol'   => $symbol,
            'apikey'   => $this->apiKey,
        ]);

        return $response->successful() ? $response->json() : null;
    }

    /**
     * Fetch daily adjusted time series.
     */
    protected function fetchDailySeries(string $symbol, bool $full = false): ?array
    {
        $response = Http::get('https://www.alphavantage.co/query', [
            'function'   => 'TIME_SERIES_DAILY_ADJUSTED',
            'symbol'     => $symbol,
            'outputsize' => $full ? 'full' : 'compact',
            'apikey'     => $this->apiKey,
        ]);

        return $response->successful()
            ? $response->json()['Time Series (Daily)'] ?? null
            : null;
    }

    /**
     * Fetch intraday time series for given interval.
     */
    protected function fetchIntradaySeries(string $symbol, string $interval, bool $full = false): ?array
    {
        $response = Http::get('https://www.alphavantage.co/query', [
            'function'   => 'TIME_SERIES_INTRADAY',
            'symbol'     => $symbol,
            'interval'   => $interval,
            'outputsize' => $full ? 'full' : 'compact',
            'apikey'     => $this->apiKey,
        ]);

        return $response->successful()
            ? $response->json()["Time Series ({$interval})"] ?? null
            : null;
    }
}