<?php

namespace App\Models;

use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\Uuid;
use MBarlow\Megaphone\HasMegaphone;
use Carbon\Carbon;

class User extends Authenticatable
{
    use HasApiTokens, Notifiable, SoftDeletes, Uuid, HasMegaphone;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    public $sortable = [
        'first_name',
        'last_name',
        'email',
        'status',
        'created_at',
    ];
    protected $fillable = [
        'first_name',
        'last_name',
        'country_id',
        'business_id',
        'email',
        'email_verify',
        'verification_code',
        'email_time',
        'ip_address',
        'password',
        'last_login',
        'fa_expiring',
        'account_type',
        'phone',
        'phone_verify',
        'phone_code',
        'phone_time',
        'otp_required',
        'token_expired',
        'social',
        'login_alert',
        'transaction_notification',
        'promotional_emails',
        'fa_status',
        'googlefa_secret',
        'contact_id',
        'social_account',
        'social_account_id',
        'status',
        'iso2',
        'mobile_code',
        'referral',
        'merchant_id',
        'ref_waivers',
        'avatar',
        'language',
        'referred_date',
        'email_auth',
    ];
    protected $guard = 'user';

    protected $table = 'users';

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password', 'remember_token',
    ];

    public function userFunds($type = null)
    {
        $account = Balance::whereUserId($this->id)->get()->sum('amount');
        $followed = Followed::whereStatus('pending')->whereUserId($this->id)->get()->sum('amount');
        return ($type == null) ? $account + $followed : (($type == 'units') ? $followed : $account);
    }

    public function followedPlan($plan = null)
    {
        return Followed::whereUserId($this->id)->orderby('id', 'DESC')
        ->when(($plan != null), function ($query) use ($plan) {
            return $query->wherePlanId($plan);
        })
        ->exists();
    }
    
    public function followed($plan = null, $type = null)
    {
        return Followed::whereUserId($this->id)->orderby('id', 'DESC')
        ->when(($plan != null), function ($query) use ($plan) {
            return $query->wherePlanId($plan);
        })
        ->when(($type != null), function ($query) use ($type) {
            return $query->whereRelation('plan', 'type', '=', $type);
        })
        ->get();
    }
    
    public function followeds()
{
    return $this->hasMany(Followed::class, 'user_id');
}
    
    public function planUnits($plan = null)
    {
        return Units::whereUserId($this->id)->whereType('buy')->orderby('created_at', 'DESC')
        ->when(($plan != null), function ($query) use ($plan) {
            return $query->wherePlanId($plan);
        })
        ->get();
    }
    
    public function planUnitsSold($plan = null)
    {
        return Units::whereUserId($this->id)->whereType('sale')->orderby('id', 'DESC')
        ->when(($plan != null), function ($query) use ($plan) {
            return $query->wherePlanId($plan);
        })
        ->get();
    }

    public function planUnitsProject($plan = null)
    {
        return Units::whereUserId($this->id)->whereMutual(0)->whereType('buy')->orderby('created_at', 'DESC')
        ->when(($plan != null), function ($query) use ($plan) {
            return $query->wherePlanId($plan);
        })
        ->get();
    }

    public function userCharges($type = null, $duration = null)
    {
        if ($duration == null) {
            $charge = ($type === null) ? Transactions::whereUserId($this->id)->whereStatus('success')->get() : Transactions::whereUserId($this->id)->whereStatus('success')->whereType($type)->get();
        } else {
            if ($duration == 'today') {
                $charge = Transactions::whereUserId($this->id)->whereStatus('success')->where('created_at', '=', Carbon::today())->get();
            } elseif ($duration == 'week') {
                $charge = Transactions::whereUserId($this->id)->whereStatus('success')->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])->get();
            } elseif ($duration == 'month') {
                $charge = Transactions::whereUserId($this->id)->whereStatus('success')->whereMonth('created_at', '=', date('m'))->whereYear('created_at', '=', date('Y'))->get();
            } elseif ($duration == 'year') {
                $charge = Transactions::whereUserId($this->id)->whereStatus('success')->whereYear('created_at', '=', date('Y'))->get();
            }
        }
        return [$charge->sum('charge'), $charge->count()];
    }

    public function devices()
    {
        return Devices::whereUser_id($this->business->user->id)->wherebusiness_id($this->business_id)->orderby('created_at', 'desc')->paginate(10);
    }

    public function ticket()
    {
        return $this->hasMany(Ticket::class, 'user_id')->whereBusinessId($this->business_id)->orderBy('created_at', 'desc');
    }    
    
    public function banks()
    {
        return $this->hasMany(UserBank::class, 'user_id')->orderBy('created_at', 'desc');
    }    
    
    public function referrals()
    {
        return $this->hasMany(User::class, 'referral')->orderBy('created_at', 'desc');
    }

    public function business()
    {
        return $this->belongsTo(Business::class, 'business_id', 'reference');
    }

    public function audit()
    {
        return $this->hasMany(Audit::class, 'user_id');
    }

    public function contact()
    {
        return $this->belongsTo(Contact::class, 'contact_id')->withTrashed();
    }

    public function businesses()
    {
        return $this->hasMany(Business::class, 'user_id');
    }

    public function getCountrySupported()
    {
        return $this->belongsTo(Countrysupported::class, 'country_id');
    }

    public function getCountry()
    {
        return Country::find($this->getCountrySupported->country_id);
    }

    public function getState()
    {
        return Shipstate::wherecountry_code($this->getCountry()->iso2)->orderby('name', 'asc')->get();
    }

    public function myState()
    {
        return Shipstate::whereid($this->state)->first();
    }

    public function getFirstBalance()
    {
        return Balance::where('user_id', $this->id)->wherebusiness_id($this->business_id)->where('country_id', $this->country_id)->first();
    }
    
    public function withdrawBalance()
    {
       return UserWithdraw::where('user_id', $this->id)->where('status', '1')->sum('amount');
    }

    public function getBalance($id)
    {
        return Balance::where('user_id', $this->id)->wherebusiness_id($this->business_id)->where('country_id', $id)->first();
    }
}
