<?php

namespace App\Models;

use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;
use App\Traits\Uuid;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Cache;
use Services\Cachable\ModelCaching\Traits\Cachable;
use Carbon\Carbon;

class Admin extends Authenticatable
{
    use Notifiable, Uuid, SoftDeletes, Cachable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'first_name',
        'last_name',
        'username',
        'password',
        'status',
        'profile',
        'promo',
        'support',
        'deposit',
        'payout',
        'news',
        'message',
        'knowledge_base',
        'email_configuration',
        'general_settings',
        'investment',
        'token',
        'token_expired',
    ];
    protected $guard = 'admin';

    protected $table = "admin";

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
    ];

    public function automatedGateway()
    {
        return Gateway::wheretype(0)->orderBy('id', 'DESC')->get();
    }

    public function manualGateway()
    {
        return Gateway::wheretype(1)->orderBy('id', 'DESC')->get();
    }

    public function allGateway()
    {
        return Gateway::all();
    }

    public function emailTemplate()
    {
        return Emailtemplate::all();
    }

    public function unreadMessages()
    {
        return Messages::whereseen(0)->count();
    }

    public function pendingPayout()
    {
        return Transactions::whereStatus('pending')->whereType('payout')->count();
    }

    public function pendingDeposit()
    {
        return Transactions::whereStatus('pending')->whereType('deposit')->orWhere('type', 'bank_transfer')->whereStatus('pending')->count();
    }

    public function currency()
    {
        return Countrysupported::whereId(1)->first();
    }

    public function socialLinks()
    {
        return Social::all();
    }

    public function openTickets()
    {
        return Ticket::wherestatus(0)->count();
    }

    public function userFunds($type = null)
    {
        $value = Cache::remember('userFunds'. $type, 600, function () use ($type) {
            $account = Balance::all()->sum('amount');
            $followed = Followed::whereStatus('pending')->get()->sum('amount');
            return ($type == null) ? $account + $followed : (($type == 'units') ? $followed : $account);
        });
        return $value;
    }

    public function userCharges($type = null, $duration = null)
    {
        $value = Cache::remember('userCharges' . $type . $duration, 600, function () use ($type, $duration) {
            if ($duration == null) {
                $charge = ($type === null) ? Transactions::whereStatus('success')->get() : Transactions::whereStatus('success')->whereType($type)->get();
            } else {
                if ($duration == 'today') {
                    $charge = Transactions::whereStatus('success')->where('created_at', '=', Carbon::today())->get();
                } elseif ($duration == 'week') {
                    $charge = Transactions::whereStatus('success')->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])->get();
                } elseif ($duration == 'month') {
                    $charge = Transactions::whereStatus('success')->whereMonth('created_at', '=', date('m'))->whereYear('created_at', '=', date('Y'))->get();
                } elseif ($duration == 'year') {
                    $charge = Transactions::whereStatus('success')->whereYear('created_at', '=', date('Y'))->get();
                }
            }
            return [$charge->sum('charge'), $charge->count()];
        });
        return $value;
    }

    public function contacts($type = null)
    {
        $value = Cache::remember('contacts' . $type, 600, function () use ($type) {
            if ($type === null) {
                $contacts = Contact::all();
            } else if ($type === 'subscribed') {
                $contacts = Contact::whereSubscribed(1)->get();
            } else if ($type === 'unsubscribed') {
                $contacts = Contact::whereSubscribed(0)->get();
            } else if ($type === 'inbox') {
                $contacts = Messages::whereSeen(0)->get();
            } else if ($type === 'open_tickets') {
                $contacts = Ticket::whereStatus(0)->get();
            }
            return $contacts;
        });
        return $value;
    }

    public function customers($type = null)
    {
        $value = Cache::remember('customers' . $type, 600, function () use ($type) {
            if ($type === null) {
                $customers = User::withTrashed()->get();
            } else if ($type === 'active') {
                $customers = User::whereStatus(0)->get();
            } else if ($type === 'blocked') {
                $customers = User::whereStatus(1)->get();
            } else if ($type === 'kyc') {
                $customers = User::whereRelation('business', 'kyc_status', '!=', 'APPROVED')->get();
            } else if ($type === 'deleted') {
                $customers = User::onlyTrashed()->get();
            }
            return $customers->count();
        });
        return $value;
    }

    public function blogDraft()
    {
        return Blog::whereStatus(0)->count();
    }

    public function deletedMessages()
    {
        return Messages::whereNotNull('deleted_at')->withTrashed()->count();
    }

    public function pages()
    {
        return Page::orderBy('id', 'desc')->get();
    }

    public function devices()
    {
        return Devices::whereUser_id($this->id)->orderby('created_at', 'desc')->paginate(10);
    }
}
