<?php

namespace App\Http\Controllers\User;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\App;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Settings;
use App\Models\Business;
use App\Models\Ticket;
use App\Models\Contact;
use App\Models\SentEmail;
use App\Models\Followed;
use App\Models\Plans;
use Illuminate\Support\Facades\Validator;
use Sonata\GoogleAuthenticator\GoogleAuthenticator;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Sonata\GoogleAuthenticator\GoogleQrUrl;
use Illuminate\Support\Facades\DB;

class UserController extends Controller
{
    protected $user;
    protected $settings;

    public function __construct()
    {
        $this->settings = Settings::find(1);
        $self = $this;
        $this->middleware(function (Request $request, $next) use ($self) {
            $self->user = auth()->guard('user')->user();
            return $next($request);
        });
    }
    
    public function index()
    {
        $id = Auth::user()->id;

$start = Carbon::now()->subDays(29)->startOfDay();
$end   = Carbon::now()->endOfDay();

$planIds = Followed::where('user_id', $id)->pluck('plan_id');

$investmentSums = DB::table('followed')
    ->where('user_id', $id)
    ->selectRaw('DATE(created_at) as date, SUM(amount) as total')
    ->whereBetween('created_at', [$start, $end])
    ->groupBy('date')
    ->orderBy('date', 'ASC')
    ->get()
    ->keyBy('date');

$profitSums = DB::table('plan_transactions')
    ->whereIn('plan_id', $planIds)
    ->where('remark', 'profit')
    ->selectRaw('DATE(created_at) as date, SUM(limit_price) as total')
    ->whereBetween('created_at', [$start, $end])
    ->groupBy('date')
    ->orderBy('date', 'ASC')
    ->get()
    ->keyBy('date');

$labels = [];
$data = [];
$profitData = [];

$period = CarbonPeriod::create($start, $end);
foreach ($period as $date) {
    $dateKey = $date->format('Y-m-d');
    $labels[] = $date->format('M d');

    $data[] = isset($investmentSums[$dateKey])
        ? (float) $investmentSums[$dateKey]->total
        : 0;

    $profitData[] = isset($profitSums[$dateKey])
        ? (float) $profitSums[$dateKey]->total
        : 0;
}

$title = "Investment Plans";

return view('user.plan.index', compact('labels', 'data', 'profitData', 'title'));
    }

    public function planDetails(Plans $plan, $type)
    {
        if(!in_array($type, ['details', 'transactions','updates', 'followers', 'portfolio'])){
            abort(403);
        }
        
         
         
        $start = Carbon::now()->subDays(29)->startOfDay();
        $end   = Carbon::now()->endOfDay();

        
        $profitSums = DB::table('plan_transactions')
            ->where('plan_id', $plan->id)
            ->where('remark', 'profit') 
            ->selectRaw('DATE(created_at) as date, SUM(limit_price) as total')
            ->whereBetween('created_at', [$start, $end])
            ->groupBy('date')
            ->orderBy('date', 'ASC')
            ->get()
            ->keyBy('date'); 

     
        $lossSums = DB::table('plan_transactions')
            ->where('plan_id', $plan->id)
            ->where('remark', 'loss') 
            ->selectRaw('DATE(created_at) as date, SUM(limit_price) as total')
            ->whereBetween('created_at', [$start, $end])
            ->groupBy('date')
            ->orderBy('date', 'ASC')
            ->get()
            ->keyBy('date');

        $labels = [];
        $profitData = [];
        $lossData = [];

      
        $period = CarbonPeriod::create($start, $end);
        foreach ($period as $date) {
            $dateKey = $date->format('Y-m-d'); 
            $labels[] = $date->format('M d'); 
        
            $profitData[] = isset($profitSums[$dateKey]) ? (float) $profitSums[$dateKey]->total : 0;
            $lossData[]   = isset($lossSums[$dateKey]) ? (float) $lossSums[$dateKey]->total : 0;
        }

        
        return view('user.plan.details', ['title' => $plan->name, 'plan' => $plan, 'type' => $type, 'labels' => $labels, 'profitData' => $profitData, 'lossData' => $lossData]);
    }

    public function lang($locale)
    {
        App::setLocale($locale);
        session()->put('locale', $locale);
        return redirect()->back();
    }

    public function checkBusiness(Request $request)
    {
        return response()->json(['st' => (Business::whereName($request->business_name)->whereAccountType('business')->count() == 0) ? 1 : 2]);
    }

    public function createPin(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'pin' => ['numeric', 'required', 'min_digits:4', 'max_digits:6'],
            ],
            [
                'pin.required' => __('Pin is required'),
                'pin.min_digits' => __('Minimum length of pin must be 4'),
                'pin.max_digits' => __('Maximum length of pin must be 6'),
            ]
        );
        if ($validator->fails()) {
            return back()->withErrors($validator->errors())->withInput();
        }
        $this->user->business->update(['pin' => Hash::make($request->pin)]);
        createAudit('Created Transfer Pin');
        return redirect()->route('user.dashboard')->with('success', 'Transfer Pin created successfully.');
    }

    public function reactivate($user)
    {
        $xx = User::withTrashed()->whereId($user)->first();
        User::whereId($xx->id)->restore();
        Business::whereId($xx->id)->restore();
        Ticket::whereId($xx->id)->restore();
        Contact::whereId($xx->id)->restore();
        SentEmail::whereId($xx->id)->restore();
        return redirect()->route('login')->with('success', 'Account restored');
    }
     
    public function profile($type)
    {
        if(!in_array($type, ['profile', 'bank', 'security', 'notifications'])){
            abort(403);
        }
        $g = new GoogleAuthenticator();
        $secret = $g->generateSecret();
        $image = GoogleQrUrl::generate($this->user->email, $secret, $this->settings->site_name);
        return view('user.profile.index', ['title' => 'Settings', 'type' => $type, 'secret' => $secret, 'image' => $image]);
    }

    public function logout()
    {
        if (Auth::guard('user')->check()) {
            $this->user->update(['fa_expiring' => Carbon::now()->subMinutes(30)]);
            Auth::guard('user')->logout();
            session()->flash('message', 'Just Logged Out!');
            return redirect()->route('login');
        } else {
            return redirect()->route('login');
        }
    }

}
